package togos.game2.jgame;

import java.awt.Color;
import java.lang.reflect.Constructor;
import java.util.Iterator;
import java.util.Vector;

import jgame.JGObject;
import jgame.JGRectangle;
import jgame.impl.YDepthComparator;
import jgame.platform.JGEngine;
import togos.game2.data.NameIdMap;
import togos.game2.world.accessor.WorldAccessor;
import togos.game2.world.behavior.Behavable;
import togos.game2.world.behavior.Behavior;
import togos.game2.world.constants.CollisionFlags;
import togos.game2.world.constants.Directions;
import togos.game2.world.definitions.Animation;
import togos.game2.world.definitions.BoundingBox;
import togos.game2.world.definitions.DirectImage;
import togos.game2.world.definitions.Icon;
import togos.game2.world.definitions.ImageOrAnimation;
import togos.game2.world.definitions.ImageSheet;
import togos.game2.world.definitions.Plane;
import togos.game2.world.definitions.RegularImageSheet;
import togos.game2.world.definitions.Section;
import togos.game2.world.definitions.SheetImage;
import togos.game2.world.definitions.Sprite;
import togos.game2.world.definitions.Tile;

public class JGameTG2Engine
{
	//// Our special sprite class
	
	/** Stationary (, resettable?) objects */
	public static String PFX_STATIC = "stat/";
	/** Anything not stationary (or not resettable?) */
	public static String PFX_DYNAMIC = "dyn/";
	public static String PFX_DYNAMIC_DUMB = "dyn/dumb/";
	public static String PFX_DYNAMIC_AUTO = "dyn/auto/";

	protected static String getObjectPrefix( Sprite s, int quadrantIndex ) {
		if( s.isStationary() && s.isResettable() ) {
			return PFX_STATIC + "q"+quadrantIndex+"/";
		}
		if( s.isAutomatic() ) {
			return PFX_DYNAMIC_AUTO;
		}
		return PFX_DYNAMIC_DUMB;
	}
	
	protected static String getSpriteObjectName( Sprite sprite, int quadrantIndex ) {
		return getObjectPrefix(sprite, quadrantIndex)+sprite.getSpriteName();
	}
	
	protected class SpriteObject extends JGObject {
		Sprite sprite;
		
		public SpriteObject( String name, Sprite sprite, int quadrantIndex ) {
			super( name, false, 0, 0, sprite.getCollisionFlags(), null );
			
			final double uw = currentPlane.getUnitWidth();
			final double uh = currentPlane.getUnitHeight();
			
			int[] qp = getQuadrantPosition(quadrantIndex);
			this.x = sprite.getX()*currentPlane.getUnitWidth() + qp[0]*sectWidth*currentPlane.getTileWidth();
			this.y = sprite.getY()*currentPlane.getUnitHeight() + qp[1]*sectHeight*currentPlane.getTileHeight();
			this.z = sprite.getZ();
			this.sprite = sprite;
			String iconName = sprite.getIconName();
			if( iconName != null ) {
				setIcon(iconName);
			}
			BoundingBox bbox = sprite.getBoundingBox();
			if( bbox != null ) {
				this.bbox = new JGRectangle( (int)(uw*bbox.getMinX()), (int)(uh*bbox.getMinY()),
					(int)(uw*(bbox.getMaxX()-bbox.getMinX())), (int)(uh*(bbox.getMaxY()-bbox.getMinY())) );
			}
			
			String shadowIconName = (String)sprite.getAttributes().get("shadow-icon-name");
			double[] shadowOffset = (double[])sprite.getAttributes().get("shadow-offset");
			if( shadowIconName != null && shadowOffset != null ) {
				Icon shadowIcon = wa.getIcon(shadowIconName);
				if( shadowIcon == null ) {
					throw new RuntimeException("Couldn't load shadow icon "+shadowIconName+" for "+name);
				}
				this.shadow = new JGObject(name+"-shadow", false, round2(x+shadowOffset[0]*uw), round2(y+shadowOffset[1]*uh), 0, shadowIcon.getImageName());
				this.shadow.imageOffsetX = shadowIcon.getOffsetX();
				this.shadow.imageOffsetY = shadowIcon.getOffsetY();
				this.shadow.depth = shadowIcon.getDepth(); 
			}
		}
		
		public void setIcon( String iconName ) {
			Icon icon = wa.getIcon( iconName );
			if( icon == null ) {
				throw new RuntimeException("Couldn't load icon "+iconName);
			}
			String imageName = icon.getImageName();
			ensureImageLoaded(imageName, true);
			this.depth = icon.getDepth();
			this.imageOffsetX = icon.getOffsetX();
			this.imageOffsetY = icon.getOffsetY();
			setGraphic(imageName);
		}
		
		/**
		 * Returns the Sprite from whence this JGObject was created.
		 * This is not kept up-to-date, but is useful for the non-changing attributes, such as name.
		 * @return
		 */
		public Sprite getSprite() {
			return sprite;
		}
		
		// Is blocked by any of these
		public int blockMask = CollisionFlags.MOBILE_SOLID|CollisionFlags.SOLID;
		// Can pass on any of these, assuming not blocked:
		public int passMask = CollisionFlags.LAND;
		
		public JGObject shadow;
		public SpriteObject( String name, boolean autoName, double x, double y, int collisionid, String imagename, int iox, int ioy, float depth, JGRectangle bbox ) {
			super( name, autoName, x, y, collisionid, imagename );
			this.imageOffsetX = iox;
			this.imageOffsetY = ioy;
			this.depth = depth;
			this.bbox = bbox;
		}
		public SpriteObject( String name, boolean autoName, double x, double y, int collisionid, String imagename, int iox, int ioy, float baseDepth, int bbx, int bby, int bbw, int bbh ) {
			this( name, autoName, x, y, collisionid, imagename, iox, ioy, baseDepth, new JGRectangle(bbx,bby,bbw,bbh) );
		}
		protected boolean isBlockedBy(int cid) {
			return (cid & blockMask) != 0;
		}
		protected boolean canPassOver(int cid) {
			return (cid & passMask) != 0;
		}
		protected boolean canPassBG(int cid) {
			return !isBlockedBy(cid) && canPassOver(cid);
		}
		public void hit_bg(int tilecid) {
			double newX = this.getLastX();
			double newY = this.getLastY();
			if( canPassBG(checkBGCollision(0,-yspeed)) ) {
				newX = this.x;
			}
			if( canPassBG(checkBGCollision(-xspeed,0)) ) {
				newY = this.y;
			}
			this.x = newX;
			this.y = newY;
		}
		public void hit(JGObject obj) {
			double newX = this.getLastX();
			double newY = this.getLastY();
			if( !isBlockedBy(checkCollision(obj.colid,0,-yspeed)) ) {
				newX = this.x;
			}
			if( !isBlockedBy(checkCollision(obj.colid,-xspeed,0)) ) {
				newY = this.y;
			}
			this.x = newX;
			this.y = newY;
		}
		protected double round2(double v) {
			return (double)(((long)v)&(-2l));
		}
		public void postUpdate() {
			if( this.shadow != null ) {
				this.shadow.x = round2(this.x);
				this.shadow.y = round2(this.y);
			}
		}
	}
	
	protected class AutoSpriteObject extends SpriteObject {
		public AutoSpriteObject( String name, Sprite sprite, int quadrantIndex ) {
			super( name, sprite, quadrantIndex );
		}
		
		protected Behavior behavior;
		public void setBehavior( Behavior b ) {
			behavior = b;
		}
		
		protected class ASOBehavable implements Behavable {
			public double getX() {  return x / currentPlane.getUnitWidth();  }
			public double getY() {  return y / currentPlane.getTileHeight();  }
			public double getZ() {  return z;  }

			public double getVelX() {  return xspeed*jge.getFrameRate()/currentPlane.getUnitWidth();  }
			public double getVelY() {  return yspeed*jge.getFrameRate()/currentPlane.getUnitHeight();  }
			public double getVelZ() {  return zspeed*jge.getFrameRate();  }

			public void say(String what, Color textColor) {
				// TODO Auto-generated method stub
				
			}
			
			public void setPosition(double x, double y, double z) {
				AutoSpriteObject.this.x = x * currentPlane.getUnitWidth();
				AutoSpriteObject.this.y = y * currentPlane.getUnitHeight();
				AutoSpriteObject.this.z = z;
			}
			
			public void setVelocity( double x, double y, double z ) {
				AutoSpriteObject.this.setSpeed(
					x*currentPlane.getUnitWidth()/jge.getFrameRate(),
					y*currentPlane.getUnitHeight()/jge.getFrameRate());
				AutoSpriteObject.this.zspeed = z/jge.getFrameRate();
			}
			
			public void setIcon( String iconName ) {
				AutoSpriteObject.this.setIcon(iconName);
			}
		}
		
		public Behavable getBehavable() {
			return new ASOBehavable();
		}
		
		public void move() {
			if( behavior != null ) {
				behavior.doStuff( jge.getGameSpeed()/jge.getFrameRate() );
			}
		}
	}
	
	protected void createSpriteObject( String objName, Sprite sprite, int quadrantIndex ) {
		try {
			String bcn = (String)sprite.getAttributes().get("behavior-class-name");
			if( bcn != null ) {
				AutoSpriteObject aso = new AutoSpriteObject( objName, sprite, quadrantIndex );
				Class bc = Class.forName(bcn);
				Constructor cons = bc.getConstructor(new Class[]{Behavable.class});
				aso.setBehavior( (Behavior)cons.newInstance(new Object[]{aso.getBehavable()}) );
			} else {
				new SpriteObject( objName, sprite, quadrantIndex );
			}
		} catch( Exception e ) {
			throw new RuntimeException(e);
		}
	}

	////	
	
	JGEngine jge;
	WorldAccessor wa;
	
	// Width and height of a single section/quadrant (same size) in tiles
	int sectWidth = 64;
	int sectHeight = 64;

	Plane currentPlane;
	NameIdMap tileIds = new NameIdMap();

	public JGameTG2Engine( JGEngine jge, WorldAccessor wa ) {
		this.jge = jge;
		this.wa = wa;
	}
	
	public int getSectWidth() {  return sectWidth;  }
	public int getSectHeight() {  return sectHeight;  }
	
	protected void ensureSheetLoaded( String sheetName ) {
		if( !jge.imageMapIsDefined(sheetName) ) {
			ImageSheet is = wa.getImageSheet(sheetName);
			if( is == null ) {
				throw new RuntimeException("Could not find definition for image sheet '"+sheetName+"'");
			} else if( !(is instanceof RegularImageSheet) ) {
				throw new RuntimeException("Only RegularImageSheets supported; got "+
					is.getClass().getName());
			}
			RegularImageSheet ris = (RegularImageSheet)is;
			jge.defineImageMap(sheetName, is.getImageDataUri(),
				ris.getFirstX(), ris.getFirstY(),
				ris.getCellWidth(), ris.getCellHeight(),
				ris.getSkipX(), ris.getSkipY());
		}
	}
	
	public void ensureImageLoaded( String imageName, boolean allowAnimation ) {
		if( !jge.imageOrAnimationIsDefined(imageName) ) {
			ImageOrAnimation ioa = wa.getImage(imageName);
			if( ioa instanceof DirectImage ) {
				DirectImage di = (DirectImage)ioa;
				jge.defineImage(imageName,
					null, 0,
					di.getImageDataUri(),
					di.getTransformString());
			} else if( ioa instanceof SheetImage ) {
				SheetImage si = (SheetImage)ioa;
				ensureSheetLoaded(si.getSheetName());
				jge.defineImage(imageName,
					null, 0,
					si.getSheetName(), si.getSheetIndex(),
					si.getTransformString(), 0, 0, 0, 0);
			} else if( allowAnimation && ioa instanceof Animation ) {
				Animation anim = (Animation)ioa;
				String[] frames = new String[anim.getFrameImageNames().size()]; 
				frames = (String[])anim.getFrameImageNames().toArray(frames);
				for( int i=frames.length-1; i>=0; --i ) {
					ensureImageLoaded( frames[i], false );
				}
				jge.defineAnimation( imageName, frames, anim.getFrameRate()/jge.getFrameRate() );
			} else {
				System.err.println("Can only load DirectImages or SheetImages" +
					(allowAnimation ? " or Animations" : "") + " as graphics; got a " +
					ioa.getClass().getName()+" for "+imageName);
			}
		}
	}
	
	protected int getTileId( String tileName ) {
		int id = tileIds.nameToId(tileName);
		if( id == -1 ) {
			id = tileIds.addNewMapping(tileName);
			Tile t = wa.getTile(tileName);
			if( t == null ) return -1;
			String imageName = t.getImageName();
			ImageOrAnimation ioa = wa.getImage(imageName);
			// @todo: allow images and tiles to be defined separately
			// so can re-use this logic for sprites:
			if( ioa instanceof DirectImage ) {
				DirectImage di = (DirectImage)ioa;
				jge.defineImage(imageName,jge.tileidToString(id),
					t.getCollisionFlags(),
					di.getImageDataUri(),
					di.getTransformString());
			} else if( ioa instanceof SheetImage ) {
				SheetImage si = (SheetImage)ioa;
				ensureSheetLoaded(si.getSheetName());
				jge.defineImage(imageName,
					jge.tileidToString(id), t.getCollisionFlags(),
					si.getSheetName(), si.getSheetIndex(),
					si.getTransformString(), 0, 0, 0, 0);
			} else {
				System.err.println("Can only use DirectImage or SheetImages for tiles; got a " +
					ioa.getClass().getName()+" for "+tileName+"/"+imageName);
			}
		}
		return id;
	}
	
	protected void initPlane( Plane p, Section s ) {
		this.currentPlane = p;
		jge.setPFSizeNoBGFill( s.getWidth()*2, s.getHeight()*2 );
		jge.setPFWrap(true, true, 0, 0);
		/*
		jge.setCanvasSettings(
			32,
			24,
			p.getTileWidth(),
			p.getTileHeight(),
			null,
			null,
			null
		);
		*/
		jge.setDepthComparator( new YDepthComparator(1.0f,(float)p.getYDepthFactor(), jge.el) );
		
		/*
		ensureImageLoaded("butterfly-e", false);
		ensureImageLoaded("butterfly-w", false);
		ensureImageLoaded("butterfly-e-flap", false);
		ensureImageLoaded("butterfly-w-flap", false);
		*/
	}
	
	//// Quadrant management:
	
	int focusedQuadrantIndex = 0;
	// Where we are focused within that section
	double focusX, focusY;
	
	protected final static int tmod( int x, int d ) {
		if( x < 0 ) return (d-((-x)%d))%d;
		return x%d;
	}
	
	protected String[] quadrantSectionNames = new String[]{"-","-","-","-"};
	
	protected int[] getQuadrantPosition( int index ) {
		return new int[]{ index&1, (index&2)>>1 };
	}
	
	protected int getQuadrantIndex( int x, int y ) {
		return ((y&1)<<1)+(x&1);
	}
		
	/** returns ratio of how far across section */
	protected double[] getFocusPositionWithinSection() {
		int px = (int)(focusX*currentPlane.getUnitWidth()/currentPlane.getTileWidth());
		int py = (int)(focusY*currentPlane.getUnitHeight()/currentPlane.getTileHeight());
		return new double[]{px/(double)sectWidth, py/(double)sectHeight};
	}
	
	protected int getSSQ( double v ) {
		if( v < 0.4 ) {
			return -1;
		} else if( v > 0.6 ) {
			return 1;
		} else {
			return 0;
		}
	}
	
	protected void depopulateObject( JGObject obj ) {
		if( obj instanceof SpriteObject ) {
			Sprite s = ((SpriteObject)obj).getSprite();
			if( !s.isResettable() ) {
			// todo: save state if object not resettable
			}
		}
		//System.err.println("Remove dynamic object "+obj.getName());
		obj.remove();
	}
	
	protected boolean objectInQuadrant( JGObject o, int quadrantIndex ) {
		int[] qxy = getQuadrantPosition(quadrantIndex);
		int qw = sectWidth*currentPlane.getTileWidth();
		int qh = sectHeight*currentPlane.getTileHeight();
		int qminx = qxy[0]*qw;
		int qmaxx = (qxy[0]+1)*qw;
		int qminy = qxy[1]*qh;
		int qmaxy = (qxy[1]+1)*qh;
		int x =  tmod( (int)o.x, qw<<1 );
		int y =  tmod( (int)o.y, qh<<1 );
		//System.err.println( x+","+y + " in "+quadrantIndex + "? ("+qminx+"-"+qmaxx+","+qminy+"-"+qmaxy+")");
		return x >= qminx && x < qmaxx && y >= qminy && y < qmaxy;
	}
	
	protected void depopulateQuadrant( int quadrantIndex ) {
		if( !"-".equals(quadrantSectionNames[quadrantIndex]) ) {
			//System.err.println("Depopulate quadrant "+quadrantIndex+" section "+quadrantSectionNames[quadrantIndex]);
			jge.removeObjects(PFX_STATIC+"q"+quadrantIndex+"/", 0);
			Vector objects = jge.getObjects(PFX_DYNAMIC, 0, true, null);
			for( int i=objects.size()-1; i>=0; --i ) {
				JGObject o = (JGObject)objects.get(i);
				if( objectInQuadrant( o, quadrantIndex ) ) {
					depopulateObject( o );
				}
			}
		}
		quadrantSectionNames[quadrantIndex] = "-";
	}
	
	protected void populateQuadrant( int quadrantIndex, String sectionName ) {
		quadrantSectionNames[quadrantIndex] = sectionName;
		if( sectionName.equals("-") ) return;
		
		int[] sxy = getQuadrantPosition(quadrantIndex);
		Section sect = wa.getSection(sectionName);
		if( sect == null ) throw new RuntimeException("Could not load section "+sectionName);

		int toffx = sxy[0]*sectWidth;
		int toffy = sxy[1]*sectHeight;
		
		int w = sect.getWidth();
		String[] tileNames = sect.getTileNames();
		for( int y=sect.getHeight()-1; y>=0; --y ) {
			for( int x=w-1; x>=0; --x ) {
				String tileName = tileNames[y*w+x];
				int tileId = getTileId(tileName);
				jge.setTileFast(toffx+x, toffy+y, tileId);
			}
		}
		for( Iterator si=sect.getSprites().iterator(); si.hasNext(); ) {
			Sprite sprite = (Sprite)si.next();
			String objName = getSpriteObjectName(sprite, quadrantIndex);
			createSpriteObject( objName, sprite, quadrantIndex );
		}
	}
	
	/**
	 * Depopulates and populates the specified quadrant with the named section.
	 * Does not check if the section replacing is the same as the one being replaced.
	 * @param quadrantIndex
	 * @param sectionName
	 */
	protected void repopulateQuadrant( int quadrantIndex, String sectionName ) {
		depopulateQuadrant(quadrantIndex);
		populateQuadrant(quadrantIndex,sectionName);
	}
	
	/**
	 * Ensures that the given quadrant has the specified section loaded.
	 * If it does not, it will depopulate whatever is there, load the named section,
	 * and populate the quadrant with its contents.
	 * @param quadrantIndex
	 * @param sectionName
	 */
	protected void ensureQuadrantSection( int quadrantIndex, String sectionName ) {
		if( !quadrantSectionNames[quadrantIndex].equals(sectionName) ) {
			repopulateQuadrant( quadrantIndex, sectionName );
		}
	}

	/**
	 * If we are focused near the edge of a quadrant,
	 * ensures that neighbor quadrants are loaded with
	 * appropriate neighbor sections.
	 */
	protected void updateNeighborQuadrants() {
		Section sect = wa.getSection(quadrantSectionNames[focusedQuadrantIndex]);
		sectWidth = sect.getWidth();
		sectHeight = sect.getHeight();
		
		// Update neighbors if needed
		int cs = focusedQuadrantIndex;
		double[] pp = getFocusPositionWithinSection();
		int nx = getSSQ(pp[0]); 
		int ny = getSSQ(pp[1]);
		if( nx != 0 ) {
			// make sure e/w is updated 
			ensureQuadrantSection( cs^1, sect.getNeighborName(Directions.xyToDirection(nx, 0)));
		}
		if( ny != 0 ) {
			//throw new RuntimeException("Uh oh, gotta update dat!");
			// make sure n/s is updated 
			ensureQuadrantSection( cs^2, sect.getNeighborName(Directions.xyToDirection(0, ny)));
		}
		if( nx != 0 && ny != 0 ) {
			// make sure kitty-corner is updated
			ensureQuadrantSection( cs^3, sect.getNeighborName(Directions.xyToDirection(nx, ny)));
		}
	}
	
	////
	
	/**
	 * Focus on a point in an already-populated quadrant
	 * @param quadrant quadrant index
	 * @param x pixels from left of quadrant to center at
	 * @param y pixels from top of quadrant to center at
	 */
	protected void focus( int quadrant, int x, int y ) {
		this.focusedQuadrantIndex = quadrant;
		this.focusX = ((double)x)/currentPlane.getUnitWidth();
		this.focusY = ((double)y)/currentPlane.getUnitHeight();
		updateNeighborQuadrants();
		int[] quadrantPosition = getQuadrantPosition( focusedQuadrantIndex );
		int vcx = quadrantPosition[0]*sectWidth*currentPlane.getTileWidth() + x;
		int vcy = quadrantPosition[1]*sectHeight*currentPlane.getTileHeight() + y;
		//System.err.println("Q"+quadrant+" ("+quadrantPosition[0]+","+quadrantPosition[1]+"), "+this.focusX+","+this.focusY+" -> "+ vcx+","+vcy);
		jge.setViewOffset( vcx, vcy, true );
	}
	
	/**
	 * Focus on a particular point in the world.
	 * Will reload plane info and repopulate the quadrants if needed.
	 * @param sectionName
	 * @param x in world units, where within section to focus
	 * @param y in world units, where within section to focus
	 */
	public void focus( String sectionName, double x, double y ) {
		Section sect = wa.getSection(sectionName);
		if( currentPlane == null || !sect.getPlaneName().equals(currentPlane.getPlaneName())) {
			Plane p = wa.getPlane(sect.getPlaneName());
			if( p == null ) {
				throw new RuntimeException("Couldn't load plane "+sect.getPlaneName());
			}
			initPlane(p, sect);
		}

		focusedQuadrantIndex = 0;
		for( int i=0; i<4; ++i ) {
			if( quadrantSectionNames[i].equals(sectionName) ) {
				focusedQuadrantIndex = i; break;
			}
		}
		ensureQuadrantSection(focusedQuadrantIndex, sectionName);

		focus( focusedQuadrantIndex,
				(int)(x*currentPlane.getUnitWidth()),
				(int)(y*currentPlane.getUnitHeight()) );
	}
	
	/**
	 * Focus on an existing JGObject, assuming it is already
	 * in a populated quadrant
	 * */
	public void focus( JGObject obj ) {
		int x = tmod( (int)obj.x, jge.pfWidth() );
		int y = tmod( (int)obj.y, jge.pfHeight() );
		int tw = currentPlane.getTileWidth();
		int th = currentPlane.getTileHeight();
		focus( getQuadrantIndex(x/tw/sectWidth, y/th/sectHeight),
				x%(sectWidth*tw),
				y%(sectHeight*th) );
	}
}
